/*
 * ledtrig-gio.c - LED Trigger Based on GPIO events
 *
 * Copyright 2009 Felipe Balbi <me@felipebalbi.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include "ledtrig-wifiled.h"
#include <linux/timer.h>

#define WIFI_OP_MAX 16
#define WIFI_OP_MAX_SHOWN 64
#define WIFILED_TRIGGER_NAME "wifi"

#define HZ_DIV (1000 / HZ);

//Note every pattern is a blink pattern, also on and off

typedef enum
{
    offState,
    onState
} ledState;

typedef enum
{
    offDisableLeds = 0,
    ondisableLeds
} powersaveState;

struct wifi_blink {
    const char      name[16];
    const char      ledName[32];
    int             on;
    int             off;
    unsigned long   currentTime;
    ledState        currentState;
    polarity_t      polarity;
    unsigned long   expiryTime;
};

struct wifi_trig_data {
	struct led_classdev *led;
    struct wifi_blink current_scheme;

    powersaveState disableLeds;
    const char operation[WIFI_OP_MAX];
};

struct  wifi_blink wps_blink = {
    .name = "wps",
    .on = 500,
    .off = 500
};

// -1: infinity, don't care
struct  wifi_blink on_blink = {
    .name = "on",
    .on = -1,
    .off = -1
};

struct  wifi_blink on_blink_disableLeds = {
    .name = "on",
    .on = 120000,
    .off = -1
};

struct  wifi_blink off_blink = {
    .name = "off",
    .on = -1,
    .off = -1
};

struct  wifi_blink update_blink = {
    .name = "update",
    .on = 100,
    .off = 100
};


static void stopTimer();
static void startTimer(void *data);


static struct timer_list blinkTimer;
static unsigned timerAlreadyStarted = 0;

/*
 * used to set wifi led to "on" for 2 minutes during bootup even when all leds are supposed to be off
 */
static bool useWiFiLedTimeout = true;
static bool firstCallSinceBootup = true;

void callbackLedBlink (unsigned long data);
void callbackLedBlink (unsigned long data) 
{
    struct led_classdev *led = (void *)data;
	struct wifi_trig_data *wifi_data = (struct wifi_trig_data*)led->trigger_data;
    unsigned long expiryTime = jiffies;
    int gpio = getGpioByName(led,led->name);
    int value_on  = (wifi_data->current_scheme.polarity == POL_HA) ? 1 : 0;
    int value_off = !value_on;
    bool startTimer = false;

    //calculate the next timeout
    if (wifi_data->current_scheme.currentState == onState)
    {
        if (wifi_data->current_scheme.off != -1)
        {
            expiryTime += wifi_data->current_scheme.off / HZ_DIV;
            startTimer = true;
        }
        //switch it off, switch state
        dvl_led_set(gpio,value_off);
        wifi_data->current_scheme.currentState = offState;
    }
    else
    {
        if (wifi_data->current_scheme.on != -1)
        {
            expiryTime += wifi_data->current_scheme.on / HZ_DIV;
            startTimer = true;
        }

        //switch it on, switch state
        dvl_led_set(gpio,value_on);
        wifi_data->current_scheme.currentState = onState;
    }

    //resetup timer
    if (startTimer)
    {
        blinkTimer.expires = expiryTime;
        add_timer (&blinkTimer);
    }
    else
    {
        timerAlreadyStarted = 0;
    }
};

static void conOperation(struct wifi_trig_data *wifi_data, const char* operation, size_t n, char *buf, size_t *offset)
{
    //safety
    if (n + *offset > WIFI_OP_MAX_SHOWN)
    {
        printk("%s: cannot show operation.\n",WIFILED_TRIGGER_NAME);
        return;
    }

    char tmpBuf[WIFI_OP_MAX] = {0};
    if (strncmp(wifi_data->operation,operation,n) == 0)
    {
        sprintf(tmpBuf, "[%s] ", operation);
        memcpy(buf+(*offset),tmpBuf,n+2+1); //2 => [] + 1 => blank
        *offset += n+2+1;
    }
    else
    {
        sprintf(tmpBuf, "%s ", operation);
        memcpy(buf+(*offset),tmpBuf,n+1);//1 => blank
        *offset += n+1;
    }
}

/*
 * sysfs operation
 */
static ssize_t wifiled_trig_operation_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct wifi_trig_data *wifi_data = led->trigger_data;

    {//TODO: stupid, make this more nice
        char out[WIFI_OP_MAX_SHOWN] = {0};
        size_t offset = 0;

        conOperation(wifi_data,"off",3,out,&offset);
        conOperation(wifi_data,"on",2,out,&offset);
        conOperation(wifi_data,"wps",3,out,&offset);

        //print to console
        return sprintf(buf,"%s\n",out);
    }
}


static void handleOnOperation(struct led_classdev *led)
{
/*
 * on disableLeds mode
 *     switch led on
 *     add timer to automaticly switch off led
 * else
 *     switch led on
 *
 * Notice:  ignore currentState. if led is on switch it on again
 *          it does not matter
 */
	struct wifi_trig_data *wifi_data = led->trigger_data;
    int gpio = getGpioByName(led,led->name);
    unsigned long expiryTime = jiffies;

    firstCallSinceBootup = false; //deactivates use of led-on-timeout when switching to leds-disabled

    //handle disableLeds
    int value_on  = (wifi_data->current_scheme.polarity == POL_HA) ? 1 : 0;

    if (wifi_data->disableLeds == offDisableLeds)
    {
        stopTimer();

        //switch it on
        dvl_led_set(gpio,value_on);
        wifi_data->current_scheme.currentState = onState;
    }
    else
    {
        if (useWiFiLedTimeout)
        {
            stopTimer();

            //switch it on
            dvl_led_set(gpio,value_on);
            wifi_data->current_scheme.currentState = onState;

            startTimer(led);

            expiryTime += wifi_data->current_scheme.on / HZ_DIV;

            blinkTimer.expires = expiryTime;
            add_timer (&blinkTimer);

            printk("%s: led on timer started.\n",WIFILED_TRIGGER_NAME);

            useWiFiLedTimeout = false;
        }
    }
}

static void handleOffOperation(struct led_classdev *led)
{
/*
 * on disableLeds mode
 *     don't care
 *
 * Notice:  ignore currentState. if led is off switch it off again
 *          it does not matter
 */
	struct wifi_trig_data *wifi_data = led->trigger_data;
    int gpio = getGpioByName(led,led->name);

    //handle disableLeds
    int value_off  = !((wifi_data->current_scheme.polarity == POL_HA) ? 1 : 0);
    dvl_led_set(gpio,value_off);
    wifi_data->current_scheme.currentState = offState;
    stopTimer();
}

static void handleWpsOperation(struct led_classdev *led)
{
/*
 * on disableLeds mode
 *     don't care
 */
	struct wifi_trig_data *wifi_data = led->trigger_data;
    int gpio = getGpioByName(led,led->name);
    unsigned long expiryTime = jiffies;

    //setup first timeout
    stopTimer();
    startTimer(led);

    if (wifi_data->current_scheme.currentState == onState)
    {
        expiryTime += wifi_data->current_scheme.off / HZ_DIV;
    }
    else
    {
        expiryTime += wifi_data->current_scheme.on / HZ_DIV;
    }

    blinkTimer.expires = expiryTime;
    add_timer (&blinkTimer);
}

static void handleUpdateOperation(struct led_classdev *led)
{
/*
 * on disableLeds mode
 *     don't care
 */
	struct wifi_trig_data *wifi_data = led->trigger_data;
    int gpio = getGpioByName(led,led->name);
    unsigned long expiryTime = jiffies;

    //setup first timeout
    stopTimer();
    startTimer(led);

    if (wifi_data->current_scheme.currentState == onState)
    {
        expiryTime += wifi_data->current_scheme.off / HZ_DIV;
    }
    else
    {
        expiryTime += wifi_data->current_scheme.on / HZ_DIV;
    }

    blinkTimer.expires = expiryTime;
    add_timer (&blinkTimer);
}


static ssize_t wifiled_trig_operation_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct wifi_trig_data *wifi_data = led->trigger_data;
    char wifi_operation[WIFI_OP_MAX] = {0};
	int ret, gpio;

    strncpy(wifi_data->operation,buf,sizeof(wifi_data->operation));

    //setup scheme
    if (!strncmp(buf,"wps",3))  
    {
        memcpy(&wifi_data->current_scheme,&wps_blink,sizeof(struct wifi_blink));
    }
    else if (!strncmp(buf,"on",2))
    {
        //setup another on state in disableLeds mode
        if (wifi_data->disableLeds == offDisableLeds)
            memcpy(&wifi_data->current_scheme,&on_blink,sizeof(struct wifi_blink));
        else
            memcpy(&wifi_data->current_scheme,&on_blink_disableLeds,sizeof(struct wifi_blink));
    }
    else if (!strncmp(buf,"off",3))
    {
        memcpy(&wifi_data->current_scheme,&off_blink,sizeof(struct wifi_blink));
        useWiFiLedTimeout = true; //reactivate timeout since led has been configured as off explicitly
    }
    else if (!strncmp(buf,"update",6))
    {
        memcpy(&wifi_data->current_scheme,&update_blink,sizeof(struct wifi_blink));
    }
    else
    {
        stopTimer();
        printk("%s: operation \"%s\" is not supported.\n",WIFILED_TRIGGER_NAME,buf);
        return -1;
    }

    //setup common scheme parameter
    strncpy(wifi_data->current_scheme.ledName,led->name,sizeof(wifi_data->current_scheme.ledName));
    gpio = getGpioByName(led,led->name);
    wifi_data->current_scheme.polarity = getGpioPolByName(led,&led->name);
    if (dvl_led_get(gpio))
    {
        if (wifi_data->current_scheme.polarity == POL_HA)
            wifi_data->current_scheme.currentState = onState;
        else
            wifi_data->current_scheme.currentState = offState;
    }
    else
    {
        if (wifi_data->current_scheme.polarity == POL_HA)
            wifi_data->current_scheme.currentState = offState;
        else
            wifi_data->current_scheme.currentState = onState;
    }

#if 0 //Debug
    printk("operation store\n");
    printk("    wifi_data->current_scheme.ledName: %s\n",wifi_data->current_scheme.ledName);
    printk("    wifi_data->current_scheme.name: %s\n",wifi_data->current_scheme.name);
    printk("    wifi_data->current_scheme.on: %i\n",wifi_data->current_scheme.on);
    printk("    wifi_data->current_scheme.off: %i\n",wifi_data->current_scheme.off);
    printk("    wifi_data->current_scheme.currentState: %i\n",wifi_data->current_scheme.currentState);
#endif

    //handle different operations
    if (!strncmp(wifi_data->current_scheme.name,"on",2))
    {
        handleOnOperation(led);
    }
    else if (!strncmp(wifi_data->current_scheme.name,"off",3))
    {
        handleOffOperation(led);
    }
    else if (!strncmp(wifi_data->current_scheme.name,"wps",3))
    {
        handleWpsOperation(led);
    }
    else if (!strncmp(wifi_data->current_scheme.name,"update",6))
    {
        handleUpdateOperation(led);
    }
    else
    {
        stopTimer();
    }

	return n;
}

static DEVICE_ATTR(operation, 0644, wifiled_trig_operation_show,
		wifiled_trig_operation_store);

/*
 * sysfs disableLeds
 */
static ssize_t wifiled_trig_disableLeds_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct wifi_trig_data *wifi_data = led->trigger_data;

    return sprintf(buf, "%u\n", wifi_data->disableLeds);
}

static ssize_t wifiled_trig_disableLeds_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct wifi_trig_data *wifi_data = led->trigger_data;

    if (strlen(buf) == 1 && (strncmp(buf,"0",1) == 0 || strncmp(buf,"1",1) == 0))
    {
        kstrtol(buf,10,&wifi_data->disableLeds);
    }
    else
    {
        printk("%s: desired disableLeds mode is not supported.\n",WIFILED_TRIGGER_NAME);
        return -1;
    }
        
    //currently only the on blink scheme has to change in disableLeds mode
    if (strncmp(wifi_data->current_scheme.name,"on",2) == 0)
    {
        if (wifi_data->disableLeds == offDisableLeds)
        {
            memcpy(&wifi_data->current_scheme,&on_blink,sizeof(struct wifi_blink));

            handleOnOperation(led);
        }
        else
        {
            memcpy(&wifi_data->current_scheme,&on_blink_disableLeds,sizeof(struct wifi_blink));

            if (firstCallSinceBootup)
            {
                useWiFiLedTimeout = true;

                handleOnOperation(led);
            }
            else
            {
                useWiFiLedTimeout = false;

                handleOffOperation(led);
            }
        }
    }

    firstCallSinceBootup = false;

    //NOTICE: the other operations are currently not affected

	return n;
}

static DEVICE_ATTR(disableLeds, 0644, wifiled_trig_disableLeds_show,
		wifiled_trig_disableLeds_store);

static void wifiled_trig_activate(struct led_classdev *led)
{
	struct wifi_trig_data *wifi_data;
	int ret, gpio;

    if (timerAlreadyStarted)
    {
        //currently there is no multi led implementation, I assume there 
        //is only one wifi led (new 2014 LED scheme), if you need a second 
        //try some kind of list to handle and distinguish between the different 
        //configs. You have also to reimplement the timer handling
        printk("%s: trigger is already in use.",WIFILED_TRIGGER_NAME);
        return;
    }

	wifi_data = kzalloc(sizeof(*wifi_data), GFP_KERNEL);
	if (!wifi_data)
		return;

	ret = device_create_file(led->dev, &dev_attr_operation);
	if (ret)
		goto err;

	ret = device_create_file(led->dev, &dev_attr_disableLeds);
	if (ret)
		goto err;

    wifi_data->led = led;
	led->trigger_data = wifi_data;

    //setup initial disableLeds state
    wifi_data->disableLeds = offDisableLeds;

    //setup initial scheme
    memcpy(&wifi_data->current_scheme,&off_blink,sizeof(struct wifi_blink));
    strncpy(wifi_data->operation,"off",3);
    gpio = getGpioByName(led,led->name);
    if (dvl_led_get(gpio))
        wifi_data->current_scheme.currentState = onState;
    else
        wifi_data->current_scheme.currentState = offState;

    return;

err:
	kfree(wifi_data);
}

static void stopTimer()
{
    if (timerAlreadyStarted)
    {
        if (!del_timer (&blinkTimer))
        {
        	printk (KERN_INFO "Couldn't remove timer!!\n");
        }
        timerAlreadyStarted = 0;
    }
}

static void startTimer(void *data)
{
    if (!timerAlreadyStarted)
    {
        //only once
        init_timer (&blinkTimer);
        blinkTimer.function = callbackLedBlink;
        blinkTimer.data = data;

        timerAlreadyStarted = 1;
    }
}


static void wifiled_trig_deactivate(struct led_classdev *led)
{
    stopTimer();

    struct wifi_trig_data *wifi_data = led->trigger_data;

	if (wifi_data) {
		device_remove_file(led->dev, &dev_attr_operation);
		kfree(wifi_data);
        led->trigger_data = NULL;
	}
}

struct led_trigger wifiled_trigger = {
	.name		= WIFILED_TRIGGER_NAME,
	.activate	= wifiled_trig_activate,
	.deactivate	= wifiled_trig_deactivate,
};
