/*
 * ledtrig-dvl.c - LED Trigger for led blinking
 *
 * Copyright 2014 devolo
 *
 */

#include "ledtrig-dvl.h"

#define DVL_GENERIC_TRIGGER_NAME "dvl_generic"


void handleChange(struct led_classdev *led);
void blink_timer_callback(unsigned long arg) ;


static void stopTimer(struct led_classdev *led)
{
    struct dvl_generic_trig_data *trig_data = led->trigger_data;

    if (!del_timer_sync (&(trig_data->blinkTimer)))
    {
      	/* printk (KERN_INFO "Couldn't remove timer!!\n");*/
    }
}

/* sysfs read from operation
 * shows all the states and the currently set in []
 */
static ssize_t operation_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct dvl_generic_trig_data *trig_data = led->trigger_data;
    char s_out[1024] = {0};
    char s_current[255] = {0};
    int i = 0;

    if (trig_data->led_states != NULL)
    {
        while(trig_data->led_states[i].valid)
        {
            if (i != trig_data->current_led_blink_state)
            {
                sprintf(s_current, "%s ", trig_data->led_states[i].name );
                strcat(s_out, s_current);
            }
            else
            {
                sprintf(s_current, "[%s] ", trig_data->led_states[i].name );
                strcat(s_out, s_current);
            }
            i++;
        }
    }
    else sprintf(s_out, "No led state table!");

    return sprintf(buf,"%s\n",s_out);
}

/*  sysfs write into operation
 *  write the name of the new state to set it
 */
static ssize_t operation_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct dvl_generic_trig_data *trig_data = led->trigger_data;
    int i = 0;
    int statefound = 0; 

    while (trig_data->led_states[i].valid)
    {
        if (strcmp(buf, trig_data->led_states[i].name) == 0)
        {
            trig_data->current_led_blink_state = i;
            statefound = 1;
        }
        i++;
    }

    if (statefound == 0)
    {
        printk("%s: not a valid state %s led %s !\n", DVL_GENERIC_TRIGGER_NAME, buf, led->name);
        return -1;
    }

    handleChange(led);

	return n;
}

/* sysfs read from disableLeds
 * shows the disabledLeds state and the current state in []
 */
static ssize_t disableLeds_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct dvl_generic_trig_data *trig_data = led->trigger_data;

    char s_out[255] = {0};

    if (trig_data->generalLEDState == enableState)
    {
        sprintf(s_out, "0 [1] ");
    }
    else
    {
        sprintf(s_out, "[0] 1 ");
    }

    return sprintf(buf, "%s\n", s_out);
}

/* sysfs write into disabledLleds
 * write the new state as 0/1 or off/on 
 */
static ssize_t disableLeds_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	struct led_classdev *led = dev_get_drvdata(dev);
	struct dvl_generic_trig_data *trig_data = led->trigger_data;

    if ((strncmp(buf,"0",1) == 0) || (strncmp(buf,"off",3) == 0))
    {
        trig_data->generalLEDState = enableState;
    }
    else if ((strncmp(buf,"1",1) == 0) || (strncmp(buf,"on",2) == 0))
    {
        trig_data->generalLEDState = disableState;
    }
    else
        printk("%s: disableLeds mode %s is not supported.\n", DVL_GENERIC_TRIGGER_NAME, buf);


    handleChange(led);

	return n;
}

/* sysfs */
static DEVICE_ATTR(disableLeds, 0644, disableLeds_show, disableLeds_store);

/* sysfs */
static DEVICE_ATTR(operation, 0644, operation_show, operation_store);

/* trigger activate funtion
 * adds the new operations for the led trigger to the filesystem */
static void dvl_generic_trig_activate(struct led_classdev *led)
{
	struct dvl_generic_trig_data *trig_data;
	int ret;

    if (isTriggerActiveOnLEDByLedName(led->name) == -1) 
    {
        printk("%s trigger not available !", DVL_GENERIC_TRIGGER_NAME);
        return;
    }

    if (isTriggerActiveOnLEDByLedName(led->name) == 1)
    {
        printk("%s trigger is already in use !", DVL_GENERIC_TRIGGER_NAME);
        return;
    }

    if (setActivateTriggerOnLEDByLedName(led->name, 1) == -1)
    {
        printk("%s not able to activate trigger !", DVL_GENERIC_TRIGGER_NAME);
        return;
    }

	trig_data = kzalloc(sizeof(*trig_data), GFP_KERNEL);
	if (!trig_data)
		return;

    init_timer(&(trig_data->blinkTimer));

	ret = device_create_file(led->dev, &dev_attr_operation);
	if (ret)
    {
        kfree(trig_data);
		return;
    }

	ret = device_create_file(led->dev, &dev_attr_disableLeds);
    if (ret)
    {
        kfree(trig_data);
        return;
    }

    trig_data->led = led;
	led->trigger_data = trig_data;  /* void ptr */

    //setup initial generalLEDState state
    trig_data->generalLEDState = enableState;
    trig_data->current_led_blink_state = 0;

    trig_data->led_states = getLedStatesByLedName(led->name);

    if (trig_data->led_states == NULL)
    {
        printk("%s: No blink state table for led %s !\n", DVL_GENERIC_TRIGGER_NAME, led->name);
        kfree(trig_data);
    }

    return;
}

static void dvl_generic_trig_deactivate(struct led_classdev *led)
{
    struct dvl_generic_trig_data *trig_data = led->trigger_data;
    stopTimer(led);

	if (trig_data) {
		device_remove_file(led->dev, &dev_attr_operation);
        device_remove_file(led->dev, &dev_attr_disableLeds);
		kfree(trig_data);
        led->trigger_data = NULL;
	}

    setActivateTriggerOnLEDByLedName(led->name, 0);
}

/* handleChange
 * handles any changed of the blink states which can be a new state or disable of LEDs.
 */
void handleChange(struct led_classdev *led)
{
	struct dvl_generic_trig_data *trig_data = led->trigger_data;
    int polarity, gpio;
    struct led_blink_state* current_led_state;
    unsigned long expiryTime;
    int value_on, value_off;
    int on_ms = 0;
    int off_ms = 0; 

    stopTimer(led);

    current_led_state = &trig_data->led_states[trig_data->current_led_blink_state];

    if (trig_data->generalLEDState == enableState)
    {
        on_ms = current_led_state->ENABLED_STATE_on_ms;
        off_ms = current_led_state->ENABLED_STATE_off_ms;
    }
    if (trig_data->generalLEDState == disableState)
    {
        on_ms = current_led_state->DISABLED_STATE_on_ms;
        off_ms = current_led_state->DISABLED_STATE_off_ms;
    }

    gpio = getGpioByName(led,led->name);
    polarity = getGpioPolByName(led,led->name);

    expiryTime = jiffies;
    value_on  = (polarity == POL_HA) ? 1 : 0;
    value_off = (polarity == POL_HA) ? 0 : 1;

    if (on_ms == -1)      /* always on */
    {
        dvl_led_set(gpio,value_on);
    }
    else if (on_ms == 0)  /* always off */
    {
        dvl_led_set(gpio,value_off);
    }
    else                  /* blink */
    {
        trig_data->timerdata.gpio = gpio;
        trig_data->timerdata.polarity = polarity;
        trig_data->timerdata.on_ms = on_ms;
        trig_data->timerdata.off_ms = off_ms;
        trig_data->timerdata.on_blink = 1;

        //switch it on to start
        dvl_led_set(gpio,value_on);

        expiryTime += (on_ms * HZ) / 1000;

        init_timer(&(trig_data->blinkTimer));
        trig_data->blinkTimer.data = (unsigned long)trig_data;
        trig_data->blinkTimer.function = blink_timer_callback;
        trig_data->blinkTimer.expires = expiryTime;
        add_timer (&trig_data->blinkTimer);

        // printk("%s: led on timer started.\n",DVL_GENERIC_TRIGGER_NAME);
    }
}

/* Timer callback function */
void blink_timer_callback(unsigned long arg) 
{
	struct dvl_generic_trig_data *trig_data = (struct dvl_generic_trig_data*)arg;
    int value_on   = (trig_data->timerdata.polarity == POL_HA) ? 1 : 0;
    int value_off  = (trig_data->timerdata.polarity == POL_HA) ? 0 : 1;
    unsigned long expiryTime = jiffies;
    
    if (trig_data->timerdata.on_blink) /* if it was on before, do the off cycle */
    {
        trig_data->timerdata.on_blink = 0;
        expiryTime += (trig_data->timerdata.off_ms * HZ) / 1000; 
        dvl_led_set(trig_data->timerdata.gpio, value_off);
    }
    else /* if it was off before, do the on cycle */
    {
        trig_data->timerdata.on_blink = 1;
        expiryTime += (trig_data->timerdata.on_ms * HZ) / 1000; 
        dvl_led_set(trig_data->timerdata.gpio, value_on);
    }

    init_timer(&(trig_data->blinkTimer));
    trig_data->blinkTimer.data = (unsigned long)trig_data;
    trig_data->blinkTimer.function = blink_timer_callback;
    trig_data->blinkTimer.expires = expiryTime;
    add_timer (&trig_data->blinkTimer);
}

struct led_trigger dvl_generic_trigger = {
	.name		= DVL_GENERIC_TRIGGER_NAME,
	.activate	= dvl_generic_trig_activate,
	.deactivate	= dvl_generic_trig_deactivate,
};
