#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/timer.h>

#include "dvltrafin_mod.h"

MODULE_LICENSE("GPL");
MODULE_AUTHOR("devolo AG, Daniel Freund");
#if defined(MAKE_ETH)
    MODULE_DESCRIPTION("Traffic indication module for ethernet devices");
#elif defined(MAKE_WIFI)
    MODULE_DESCRIPTION("Traffic indication module for wireless devices");
#endif

static char *devname = 0;
module_param(devname, charp, 0);
MODULE_PARM_DESC(devname, "Set the interface to monitor for traffic");


//####################################
//typedefs

typedef enum {
	eLED_STATE_ON, //led on
	eLED_STATE_OFF, //led off
	eLED_STATE_NOCHANGE //preserve current led state
} led_state_t;

/**
 * entry of a led blink scheme
 *
 * @state: led on or off
 * @duration: in ms, how long the state shall be preserved
 */
typedef struct {
	led_state_t state;
	unsigned int duration;
} led_blink_scheme_entry_t;


//####################################
//variables

#ifndef _DVL_CLASSIC_LED_BLINK_FREQ
//else branch further down

//check every xyz ms for link status and traffic
#ifdef TRAFFIC_CHECK_PERIOD
#undef TRAFFIC_CHECK_PERIOD
#endif
#define TRAFFIC_CHECK_PERIOD 370 //in milliseconds

static const unsigned int cTimeoutTrafficPoll = TRAFFIC_CHECK_PERIOD*HZ/1000;

//blink once per check period
static const led_blink_scheme_entry_t ledSchemeTrafficLow[] = {
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_NOCHANGE, 0} //list terminator
};

//blink twice per check period
static const led_blink_scheme_entry_t ledSchemeTrafficMed[] = {
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_NOCHANGE, 0} //list terminator
};

//blink three times per check period
static const led_blink_scheme_entry_t ledSchemeTrafficHi[] = {
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
    {eLED_STATE_OFF, 60*HZ/1000},
    {eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_NOCHANGE, 0} //list terminator
};

#else
//i.e. _DVL_CLASSIC_LED_BLINK_FREQ is defined

//check every xyz ms for link status and traffic
#ifdef TRAFFIC_CHECK_PERIOD
#undef TRAFFIC_CHECK_PERIOD
#endif
#define TRAFFIC_CHECK_PERIOD 370 //in milliseconds

static const unsigned int cTimeoutTrafficPoll = TRAFFIC_CHECK_PERIOD*HZ/1000;

//blink in 60ms interval as usually specified in devolo PIBs
static const led_blink_scheme_entry_t ledSchemeTrafficClassic[] = {
	{eLED_STATE_OFF, 60*HZ/1000},
	{eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_OFF, 60*HZ/1000},
	{eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_OFF, 60*HZ/1000},
	{eLED_STATE_ON, 60*HZ/1000},
	{eLED_STATE_NOCHANGE, 0} //list terminator
};

#endif //_DVL_CLASSIC_LED_BLINK_FREQ

static struct timer_list trafficPollTimer, trafficLedTimer;
static led_state_t currentLedState = eLED_STATE_OFF;
static unsigned int ledStateCounter = 0;
static BOOL moduleToBeUnloaded = FALSE;

#ifdef MAKE_WIFI
    static wireless_band_t lastWirelessBand = eWBAND_2G;
#endif


//####################################
//local prototypes

static void startTrafficPollTimer(void);
static void startTrafficLedTimer(const led_blink_scheme_entry_t *blinkScheme);
static BOOL stopTrafficPollTimer(void);
static BOOL stopTrafficLedTimer(void);
static void timerHandlerCheckForTraffic(unsigned long unused);

static BOOL initTimers(void);
static BOOL startTimers(void);
static BOOL stopTimers(void);

static void setLedState(led_state_t state);

#if defined(MAKE_ETH)
    static void setLanLedState(led_state_t state);
#elif defined(MAKE_WIFI)
    static void setWirelessLedState(led_state_t state);
#endif

//####################################
//external prototypes



//####################################
//local function definitions

#if defined(MAKE_ETH)
    static void setLanLedState(led_state_t state)
    {
        if (state == currentLedState)
        { //do not set same led state twice
            return;
        }

        switch (state)
        {
            case eLED_STATE_ON:
            {
                set_lan_led(1);
                break;
            }
            case eLED_STATE_OFF:
            {
                set_lan_led(0);
                break;
            }
            default:
                break;
        }

        currentLedState = state;
    }
#elif defined(MAKE_WIFI)
    static void setWirelessLedState(led_state_t state)
    {
        wireless_band_t currentWirelessBand;

        if (state == currentLedState)
        { //do not set same led state twice
            return;
        }

        currentWirelessBand = get_wireless_band();
        //if band changed since last led flicker, set the other led to off
        if (lastWirelessBand != currentWirelessBand)
        {
            lastWirelessBand = currentWirelessBand;

            if (currentWirelessBand == eWBAND_2G)
            {
                set_wireless_5g_led(0);
            }
            else
            {
                set_wireless_2g_led(0);
            }
        }

        switch (state)
        {
            case eLED_STATE_ON:
            {
                if (currentWirelessBand == eWBAND_2G)
                {
                    set_wireless_2g_led(1);
                }
                else
                {
                    set_wireless_5g_led(1);
                }
                break;
            }
            case eLED_STATE_OFF:
            {
                if (currentWirelessBand == eWBAND_2G)
                {
                    set_wireless_2g_led(0);
                }
                else
                {
                    set_wireless_5g_led(0);
                }
                break;
            }
            default:
                break;
        }

        currentLedState = state;
    }
#endif

static void setLedState(led_state_t state)
{
#if defined(MAKE_ETH)
    setLanLedState((unsigned int)(state));
#elif defined(MAKE_WIFI)
    setWirelessLedState((unsigned int)(state));
#endif
}

static void timerHandlerCheckForTraffic(unsigned long unused)
{
	const led_blink_scheme_entry_t *blinkScheme;
	traffic_intensity_t trafInt = eTRAFFIC_NONE;

	del_timer_sync(&trafficLedTimer);

    if (connection_active())
    {
        if ((trafInt = get_traffic_intensity()) != eTRAFFIC_NONE)
        {

#ifndef _DVL_CLASSIC_LED_BLINK_FREQ

            //select led blink scheme according to traffic intensity
            switch (trafInt)
            {
                case eTRAFFIC_LOW:
                    blinkScheme = ledSchemeTrafficLow;
                    break;
                case eTRAFFIC_MEDIUM:
                    blinkScheme = ledSchemeTrafficMed;
                    break;
                case eTRAFFIC_HIGH:
                    blinkScheme = ledSchemeTrafficHi;
                    break;
                default:
                    //should not occur
                    return;
                    break;
            }

#else

            blinkScheme = ledSchemeTrafficClassic;

#endif

            ledStateCounter = 0;
            setLedState((unsigned int)(blinkScheme[0].state));
            startTrafficLedTimer(blinkScheme);
        }
        else
        {
            setLedState(eLED_STATE_ON);
        }
    }
    else
    {
        setLedState(eLED_STATE_OFF);
    }

	startTrafficPollTimer();
}

static void timerHandlerToggleLed(unsigned long bs)
{
	led_blink_scheme_entry_t *blinkScheme = (led_blink_scheme_entry_t *)bs;

	if (blinkScheme[ledStateCounter].duration > 0)
	{
		setLedState(blinkScheme[ledStateCounter].state);
		ledStateCounter++;
		startTrafficLedTimer(blinkScheme);
	}
}

/**
 * on firing, check if there was some traffic
 */
static void startTrafficPollTimer(void)
{
	if (moduleToBeUnloaded)
	{
		return;
	}

	trafficPollTimer.expires = jiffies + cTimeoutTrafficPoll;
	add_timer(&trafficPollTimer);
}

static void startTrafficLedTimer(const led_blink_scheme_entry_t *blinkScheme)
{
	if (moduleToBeUnloaded)
	{
		return;
	}

	if (blinkScheme[ledStateCounter].duration > 0)
	{
		trafficLedTimer.expires = jiffies + blinkScheme[ledStateCounter].duration;
		trafficLedTimer.data = (unsigned long) blinkScheme;
		add_timer(&trafficLedTimer);
	}
}

static BOOL stopTrafficPollTimer(void)
{
	del_timer_sync(&trafficPollTimer);

	return TRUE;
}

static BOOL stopTrafficLedTimer(void)
{
	del_timer_sync(&trafficLedTimer);

	return TRUE;
}

static BOOL startTimers(void)
{
	startTrafficPollTimer();

	return TRUE;
}

static BOOL stopTimers(void)
{
	stopTrafficPollTimer();
	stopTrafficLedTimer();

	return TRUE;
}

static BOOL initTimers(void)
{
    init_timer(&trafficPollTimer);
    trafficPollTimer.function = timerHandlerCheckForTraffic;

    init_timer(&trafficLedTimer);
    trafficLedTimer.function = timerHandlerToggleLed;

	return TRUE;
}

static BOOL initLeds(void)
{
    setLedState(eLED_STATE_OFF);

    return TRUE;
}

static int __init moduleLoad(void)
{
	if (switchctrl_load(devname) && initLeds() && initTimers() && startTimers())
	{
#if defined(MAKE_ETH)
		printk(KERN_INFO "Started ethernet traffic indication module. \n");
#elif defined(MAKE_WIFI)
		printk(KERN_INFO "Started wireless traffic indication module. \n");
#endif
		return 0;
	}
	else
	{
#if defined(MAKE_ETH)
		printk(KERN_ERR "Could not start ethernet traffic indication module! \n");
#elif defined(MAKE_WIFI)
		printk(KERN_ERR "Could not start wireless traffic indication module! \n");
#endif
		return -EAGAIN;
	}
}

static void __exit moduleUnload(void)
{
	moduleToBeUnloaded = TRUE;

	setLedState(eLED_STATE_OFF);

	if (switchctrl_unload() && stopTimers())
	{
#if defined(MAKE_ETH)
		printk(KERN_INFO "Stopped ethernet traffic indication module. \n");
#elif defined(MAKE_WIFI)
		printk(KERN_INFO "Stopped wireless traffic indication module. \n");
#endif
	}
	else
	{
#if defined(MAKE_ETH)
		printk(KERN_ERR "Could not stop ethernet traffic indication module! \n");
#elif defined(MAKE_WIFI)
		printk(KERN_ERR "Could not stop wireless traffic indication module! \n");
#endif
	}
}

module_init(moduleLoad);
module_exit(moduleUnload);
