#include "memtest.h"

/**********************************************************************
 *
 * Filename:    memtest.c
 * 
 * Description: General-purpose memory testing functions.
 *
 * Notes:       This software can be easily ported to systems with
 *              different data bus widths by redefining 'datum'.
 *
 * 
 * Copyright (c) 1998 by Michael Barr.  This software is placed into
 * the public domain and may be used for any purpose.  However, this
 * notice must not be changed or removed and no warranty is either
 * expressed or implied by its publication or distribution.
 **********************************************************************/

enum {
	RET_FAIL = -1,
	RET_CONT = 0,
	RET_BREAK = 1,
};

#ifdef COMPRESSED_UBOOT
#	define prmsg(...)
#else
#	define prmsg	printf
#endif

/**********************************************************************
 *
 * Function:    memTestDataBus()
 *
 * Description: Test the data bus wiring in a memory region by
 *              performing a walking 1's test at a fixed address
 *              within that region.  The address (and hence the
 *              memory region) is selected by the caller.
 *
 * Notes:
 *
 * Returns:     0 if the test succeeds.
 *              A non-zero result otherwise.
 *
 **********************************************************************/
int memTestDataBus(volatile datum_t * address)
{
	datum_t pattern;

	prmsg("Testing data bus ...");

    /*
     * Perform a walking 1's test at the given address.
     */
    for (pattern = 1; pattern != 0; pattern <<= 1)
    {
        /*
         * Write the test pattern.
         */
        *address = pattern;

        /*
         * Write somewhere else to change bus charge
         */
        *(address+1) = ~pattern;

        /*
         * Read it back
         */
        if (*address != pattern) 
        {
        	prmsg(" failed!\n");
        	prmsg("Wrote 0x%08x - read 0x%08x.\n", pattern, *address);
            return RET_FAIL;
        }
    }

    prmsg(" done.\n");

    return (RET_CONT);

}   /* memTestDataBus() */


/**********************************************************************
 *
 * Function:    memTestAddressBus()
 *
 * Description: Test the address bus wiring in a memory region by
 *              performing a walking 1's test on the relevant bits
 *              of the address and checking for aliasing. This test
 *              will find single-bit address failures such as stuck
 *              -high, stuck-low, and shorted pins.  The base address
 *              and size of the region are selected by the caller.
 *
 * Notes:       For best results, the selected base address should
 *              have enough LSB 0's to guarantee single address bit
 *              changes.  For example, to test a 64-Kbyte region, 
 *              select a base address on a 64-Kbyte boundary.  Also, 
 *              select the region size as a power-of-two--if at all 
 *              possible.
 *
 * Returns:     0 if the test succeeds.
 *              A non-zero result otherwise.
 *
 **********************************************************************/
int memTestAddressBus(volatile datum_t * baseAddress, unsigned long nBytes)
{
    unsigned long addressMask = (nBytes/sizeof(datum_t) - 1);
    unsigned long offset;
    unsigned long testOffset;

    datum_t pattern     = (datum_t) 0xAAAAAAAA;
    datum_t antipattern = (datum_t) 0x55555555;

    prmsg("Testing address bus ...\n");

    /*
     * Write the default pattern at each of the power-of-two offsets.
     */
    prmsg("\tWriting test pattern ...");
    for (offset = 1; (offset & addressMask) != 0; offset <<= 1)
    {
        baseAddress[offset] = pattern;
    }
    prmsg(" done.\n");

    /* 
     * Check for address bits stuck high.
     */
    prmsg("\tChecking for bits stuck high ...");
    testOffset = 0;
    baseAddress[testOffset] = antipattern;

    for (offset = 1; (offset & addressMask) != 0; offset <<= 1)
    {
        if (baseAddress[offset] != pattern)
        {
        	prmsg(" failed at address 0x%08x!\n", &baseAddress[offset]);
        	prmsg("Wrote 0x%8x - read 0x%08x.\n", pattern, baseAddress[offset]);
            return RET_FAIL;
        }
    }
    prmsg(" done.\n");

    baseAddress[testOffset] = pattern;

    /*
     * Check for address bits stuck low or shorted.
     */
    prmsg("\tChecking for bits stuck low or shorted ...");
    for (testOffset = 1; (testOffset & addressMask) != 0; testOffset <<= 1)
    {
        baseAddress[testOffset] = antipattern;

		if (baseAddress[0] != pattern)
		{
			prmsg(" failed at address 0x%08x!\n", &baseAddress[0]);
			prmsg("Wrote 0x%8x - read 0x%08x.\n", pattern, baseAddress[0]);
			return RET_FAIL;
		}

        for (offset = 1; (offset & addressMask) != 0; offset <<= 1)
        {
            if ((baseAddress[offset] != pattern) && (offset != testOffset))
            {
            	prmsg(" failed at address 0x%08x!\n", &baseAddress[offset]);
				prmsg("Wrote 0x%8x - read 0x%08x.\n", pattern, baseAddress[offset]);
                return RET_FAIL;
            }
        }

        baseAddress[testOffset] = pattern;
    }
    prmsg(" done.\n"); //pattern

    prmsg("... done.\n"); //address bus test

    return RET_CONT;

}   /* memTestAddressBus() */

/**********************************************************************
 *
 * Function:    memTestStrobes()
 *
 * Description: Test bytewise write and read to check for swapped
 * 				strobe/mask lines. Base address is only needed to
 * 				write to a valid location in ram.
 *
 * Notes:
 *
 * Returns:     0 if the test succeeds.
 *
 *              A non-zero result otherwise.
 *
 **********************************************************************/
int memTestStrobes(volatile datum_t * baseAddress, unsigned int databuswidth)
{
	prmsg("Testing strobe/mask lines ...\n");

	switch (databuswidth)
	{
	case 8:
		return RET_CONT; //if buswidth is only 8 bits, then there are no strobe lines
		break;
	case 16:
		//write each byte seperately
		(*((unsigned char *)baseAddress)) = 0;
		(*(((unsigned char *)baseAddress)+1)) = 0;
		(*((unsigned char *)baseAddress)) = 0x11;
		(*(((unsigned char *)baseAddress)+1)) = 0x22;

		//read with full buswidth
		if (*baseAddress != 0x1122)
		{
			prmsg(" failed! (read 0x%4x, should be 0x1122)\n", *baseAddress);
			return RET_FAIL;
		}
		break;
	case 32:
		(*((unsigned char *)baseAddress)) = 0;
		(*(((unsigned char *)baseAddress)+1)) = 0;
		(*(((unsigned char *)baseAddress)+2)) = 0;
		(*(((unsigned char *)baseAddress)+3)) = 0;
		(*((unsigned char *)baseAddress)) = 0x11;
		(*(((unsigned char *)baseAddress)+1)) = 0x22;
		(*(((unsigned char *)baseAddress)+2)) = 0x44;
		(*(((unsigned char *)baseAddress)+3)) = 0x88;

		if (*baseAddress != 0x11224488)
		{
			prmsg(" failed! (read 0x%8x, should be 0x11224488)\n", *baseAddress);
			return RET_FAIL;
		}
		break;
	default: //unknown buswidth
		prmsg(" failed! (unsupported buswidth)\n");
		return RET_FAIL;
		break;
	}

	prmsg("... done.\n");

	return RET_CONT;
}


/**********************************************************************
 *
 * Function:    memTestDevice()
 *
 * Description: Test the integrity of a physical memory device by
 *              performing an increment/decrement test over the
 *              entire region.  In the process every storage bit 
 *              in the device is tested as a zero and a one.  The
 *              base address and the size of the region are
 *              selected by the caller.
 *
 * Notes:       
 *
 * Returns:     0 if the test succeeds.
 *
 *              A non-zero result otherwise.
 *
 **********************************************************************/
int memTestDevice(volatile datum_t * baseAddress, unsigned long nBytes)
{
    unsigned long offset;
    unsigned long nWords = nBytes / sizeof(datum_t);

    datum_t pattern;
    datum_t antipattern;

    prmsg("Testing memory devices ...\n");

    /*
     * Fill memory with a known pattern.
     */
    prmsg("\tFilling memory with test pattern ...");
    for (pattern = 1, offset = 0; offset < nWords; pattern++, offset++)
    {
        baseAddress[offset] = pattern;
    }
    prmsg(" done.\n");

    if (check_for_abort() != RET_CONT)
    	return RET_BREAK;

    /*
     * Check each location and invert it for the second pass.
     */
    prmsg("\tChecking for test pattern and inverting it ...");
    for (pattern = 1, offset = 0; offset < nWords; pattern++, offset++)
    {
    	if (check_for_abort() != RET_CONT)
    		return RET_BREAK;

        if (baseAddress[offset] != pattern)
        {
        	prmsg(" failed at address 0x%08x!\n", &baseAddress[offset]);
        	prmsg("Wrote 0x%8x - read 0x%08x.\n", pattern, baseAddress[offset]);
            return RET_FAIL;
        }

        antipattern = ~pattern;
        baseAddress[offset] = antipattern;
    }
    prmsg(" done.\n");

	if (check_for_abort() != RET_CONT)
		return RET_BREAK;

    /*
     * Check each location for the inverted pattern and zero it.
     */
    prmsg("\tChecking for inverted test pattern ...");
    for (pattern = 1, offset = 0; offset < nWords; pattern++, offset++)
    {
    	if (check_for_abort() != RET_CONT)
    		return RET_BREAK;

        antipattern = ~pattern;
        if (baseAddress[offset] != antipattern)
        {
        	prmsg(" failed at address 0x%08x!\n", &baseAddress[offset]);
			prmsg("Wrote 0x%8x - read 0x%08x.\n", antipattern, baseAddress[offset]);
            return RET_FAIL;
        }
    }
    prmsg(" done.\n");

    prmsg("... done.\n");

    return RET_CONT;

}   /* memTestDevice() */

/**********************************************************************
 *
 * Function:    check_for_abort()
 *
 * Description: Check a defined gpio line for activation
 *
 * Notes:
 *
 * Returns:     1 if the selected gpio is active
				(i.e. a button is pressed).
 *
 *              0 if the selected gpio is not active.
 *
 **********************************************************************/
int check_for_abort(void)
{
	volatile unsigned long *gpio_in_reg;
	gpio_in_reg = REG_GPIO_IN;

	if ((*gpio_in_reg & GPIO_ABORT_MEMTEST) == GPIO_ABORT_MEMTEST_ACTIVE_STATE)
	{
		prmsg("\n>>> MANUAL BREAK REQUESTED.<<<\n");
		return RET_BREAK;
	}

	return RET_CONT;
}

/**********************************************************************
 *
 * Function:    run_memtest()
 *
 * Description: call the upper tests in a certain order. start with
 * 				databus tests followed by the address bus test.
 * 				the device test is repeated indefinetely until user
 * 				request (through gpio acitvation).
 *
 * Notes:       this function must be called from the u-boot initialization
 * 				routine before(!) relocation to ram!
 *
 * Returns:     0 if the test was successful or user requested abort.
 *
 * 				-1 if the test failed at some point
 *
 **********************************************************************/
int run_memtest (datum_t* baseAddress, int numBytes)
{
	int retval = RET_CONT;

	prmsg("\n########################################\n");
	prmsg("Starting memtest. Testing %d bytes from address 0x%08x.\n", numBytes, baseAddress);
	prmsg("\n>>> PRESS SOFT-RESET BUTTON TO ABORT. <<<\n\n");

	retval = check_for_abort();

	//DataBus:
	if (retval == RET_CONT)
		retval = memTestDataBus(baseAddress);

	if (retval == RET_CONT)
		retval = check_for_abort();

	//AddressBus:
	if (retval == RET_CONT)
		retval = memTestAddressBus(baseAddress, numBytes);

	if (retval == RET_CONT)
		retval = check_for_abort();
#if 0
	//StrobeLines:
	if (retval == RET_CONT)
		retval = memTestStrobes(baseAddress, 32); //TODO: chose correct buswidth

	if (retval == RET_CONT)
		retval = check_for_abort();
#endif

#if 1
	//run device test until manual abort
	while (retval == RET_CONT)
	{
		//Memory cells:
		retval = memTestDevice(baseAddress, numBytes);

		if (retval != RET_CONT)
		{
			switch (retval) {
				case RET_BREAK:
					prmsg("Memtest aborted on user request.\n");
					break;
				case RET_FAIL:
					prmsg("Memtest FAILED!\n");
#if 1 //continue memtest even on errors until key press
					retval = RET_CONT;
#endif
					break;
				default:
					prmsg("Memtest aborted with UNKNOWN RESULT!\n");
					retval = RET_FAIL;
					break;
			}

			if (retval != RET_CONT)
			{
			    break; //exit while loop. necessary on user abort and successful pass!
			}
		}
	}
#else
	//run device test only once
	if (retval == RET_CONT)
	    retval = memTestDevice(baseAddress, numBytes);

    if (retval != RET_CONT)
    {
        switch (retval) {
            case RET_BREAK:
                prmsg("Memtest aborted on user request.\n");
                break;
            case RET_FAIL:
                prmsg("Memtest FAILED!\n");
                break;
            default:
                prmsg("Memtest aborted with UNKNOWN RESULT!\n");
                retval = RET_FAIL;
                break;
        }
    }
    else
    {
        prmsg("Memtest PASSED without errors.\n");
    }
#endif

#if 0 //loop endlessly (until button is pressed) on memtest failure
    if (retval != RET_CONT)
    {
        prmsg("\nWaiting for reset button ...\n");
        while (check_for_abort() == RET_CONT)
        {
            sleep(1);
        }
    }
#endif

	prmsg("########################################\n\n");

	return retval;
}

